//******************************************
// file			: ADC.c
// version		: V1.0
// brief		: ADC related functions
// note			: ADC related functions are gathered in this subroutine
//******************************************
#include "MQ6811.h"
#include "ADC.h"


//*********************************************************************
// name			: GetVref()
// brief		: get the value of internal ADC reference voltage (unit: mV)
// note			:
//*********************************************************************
uint GetVref(void)
{
	TYPE_WORD ADC_READ;
//	Vref2V, value is ( Real_2V )
	__asm("LD A,(0x7E64)");		// low byte
	__asm("LD (_vref_l),A");
	__asm("LD A,(0x7E63)");		// high byte
	__asm("LD (_vref_h),A");

	ADC_READ.byte[0] = vref_l;
	ADC_READ.byte[1] = vref_h & 0x0f;

	if( ADC_READ.word > 2100 )	ADC_READ.word = 2000;	// if Vref is out of range, do not use this value
	if( ADC_READ.word < 1900 )	ADC_READ.word = 2000;	// if Vref is out of range, do not use this value
	VrefReal2048 = (ulong)ADC_READ.word * 2048 / 2000;
	return ADC_READ.word;
}


//*********************************************************************
// name			: ADC_detect()
// brief		: 32 conversion times and get the sum of results
// note			: 
//*********************************************************************
ulong ADC_detect(uchar VREF, uchar SAIN, uchar ACK, uchar AMD)
{
	char  i;
	ulong adc_data = 0;
	TYPE_WORD ADC_READ;

//	ADC Config set ......
	ADCVRF.byte  = VREF;
	ADCCR1.byte  = SAIN;	// 1. set analog input channel
	ADCCR1.byte |= 16;		// 2. set analog input enable
	ADCCR1.byte |= AMD*32;	// 3. set AD operation mode
	ADCCR2.byte  = ACK;

//	ADC Single mode ......
	if( AMD == ADC_Single )
	{
		for( i=0; i<ad_test_times; i++ )
		{
			ADCCR1.bit.b7 = 1;			// start ADC sampling
			while(  ADCCR2.bit.b6 ){}	// wait for AD conversion being halted
			while( !ADCCR2.bit.b7 ){}	// wait for end of conversion
			ADC_READ.byte[0] = ADCDRL.byte;
			ADC_READ.byte[1] = ADCDRH.byte;
			adc_data += ADC_READ.word;
		}
	}
//	ADC Repeat mode ......
	else
	{
		ADCCR1.bit.b7 = 1;				// start ADC sampling
		for( i=0; i<ad_test_times; i++ )
		{
			while( !ADCCR2.bit.b7 ){}	// wait for end of conversion
			ADC_READ.byte[0] = ADCDRL.byte;
			ADC_READ.byte[1] = ADCDRH.byte;
			adc_data += ADC_READ.word;
		}
		ADCCR1.byte = 0x00;				// stop ADC sampling
	}

	return adc_data;
}


//*********************************************************************
// name			: ADC_ConvCode()
// brief		:
// note			: AD conversion result code
// 			: fcgck=16M... Conv256CLK(16us) ~ Conv1024CLK( 64us)
// 			: fcgck= 8M... Conv128CLK(16us) ~ Conv1024CLK(128us)
// 			: fcgck= 4M... Conv64CLK (16us) ~ Conv512CLK (128us)
// 			: fcgck= 2M... Conv32CLK (16us) ~ Conv256CLK (128us)
//*********************************************************************
ulong ADC_ConvCode(uchar VREF, uchar SAIN, uchar ACK, uchar AMD)
{
	if( VREF == Vref2 )
	{
		adc_value = ADC_detect(VREF,    SAIN, ACK, AMD);
		adc_GND   = ADC_detect(VREF, AIN_GND, ACK, AMD);	// internal offset

	//	subtract the ADC offset, compare the ADC result first, if the ADC result is less than the offset, then take the ADC result as 0
	//	AD conversion result code
	//	if( adc_value > adc_GND )	adc_value = ((adc_value - adc_GND)*iReal_Vref2V) / 2000 / ad_test_times;
		if( adc_value > adc_GND )	adc_value = ((adc_value - adc_GND)*VrefReal2048) / 2048 / ad_test_times;
		else						adc_value = 0;
	}
	else
	{
		adc_value = ADC_detect(VREF, SAIN, ACK, AMD) / ad_test_times;
	}

	return adc_value;
}

//*********************************************************************
// name			: ADC_ConvVolt()
// brief		:
// note			: AD conversion result (unit: mV)
// 			: fcgck=16M... Conv256CLK(16us) ~ Conv1024CLK( 64us)
// 			: fcgck= 8M... Conv128CLK(16us) ~ Conv1024CLK(128us)
// 			: fcgck= 4M... Conv64CLK (16us) ~ Conv512CLK (128us)
// 			: fcgck= 2M... Conv32CLK (16us) ~ Conv256CLK (128us)
//*********************************************************************
ulong ADC_ConvVolt(uchar VREF, uchar SAIN, uchar ACK, uchar AMD)
{
	if( VREF == Vref2 )
	{
		adc_value = ADC_detect(VREF,    SAIN, ACK, AMD);
		adc_GND   = ADC_detect(VREF, AIN_GND, ACK, AMD);	// internal offset

	//	subtract the ADC offset, compare the ADC result first, if the ADC result is less than the offset, then take the ADC result as 0
	//	convert code to voltage
		if( adc_value > adc_GND )	adc_value = ((adc_value - adc_GND)*iReal_Vref2V) / 1024 / ad_test_times;
		else						adc_value = 0;

		return adc_value;
	}

	return 9999;	// for not 2V reference voltage
}














